package com.sensetime.uc.proxy;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

@Service
public class ApiService {

    private Map<String,UserTokenResultResponse> tokenMap = new HashMap<>();
    private RestTemplate restTemplate = new RestTemplate();

    private String currToken = "curr";

    @Value("${api.base.url}")
    private String baseUrl;

    @Value("${api.access.token.endpoint}")
    private String accessTokenEndpoint;

    @Value("${api.refresh.token.endpoint}")
    private String refreshTokenEndpoint;

    @Value("${api.app.id}")
    private String appId;

    @Value("${api.app.key}")
    private String appKey;

    public ResultData<UserTokenResultResponseDTO> getAccessToken() {
        if(tokenMap.get(currToken) != null){
            UserTokenResultResponse userTokenResultResponse = tokenMap.get(currToken);
            UserTokenResultResponseDTO userTokenResultResponseDTO = genUserTokenResultResponseDTO(userTokenResultResponse);
            userTokenResultResponseDTO.setExpiresIn(userTokenResultResponseDTO.calcTokenExpiresIn());
            System.out.println("return token from local cache!");
            return ResultData.success(userTokenResultResponseDTO);
        }
        // 构建URL
        String url = baseUrl + accessTokenEndpoint;

        // 获取当前时间戳
        String timestamp = String.valueOf(Instant.now().getEpochSecond());

        // 生成签名
        String sign = generateSign(appId, appKey, timestamp);

        // 构建请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 构建请求体
        String requestJson = String.format("{\"appId\":\"%s\",\"timestamp\":\"%s\",\"sign\":\"%s\",\"grantType\":\"sign\"}", appId, timestamp, sign);

        // 构建请求体
        HttpEntity<String> entity = new HttpEntity<>(requestJson, headers);

        // 发送请求
        ResponseEntity<ResultData<UserTokenResultResponse>> response = restTemplate.exchange(url, HttpMethod.POST, entity,
                new ParameterizedTypeReference<ResultData<UserTokenResultResponse>>() {});
        UserTokenResultResponse curr = response.getBody().getData();
        tokenMap.put(currToken, curr);

        UserTokenResultResponseDTO dto = genUserTokenResultResponseDTO(curr);
        return  ResultData.success(dto);
    }

    private UserTokenResultResponseDTO genUserTokenResultResponseDTO(UserTokenResultResponse curr) {
        UserTokenResultResponseDTO dto = new UserTokenResultResponseDTO();
        dto.setAccessToken(curr.getAccessToken());
        dto.setExpiresTime(curr.getExpiresTime());
        dto.setExpiresIn(curr.getExpiresIn());
        return dto;
    }


    /**
     * 获取MD5加密的基础十六进制字符串，条件格式：appId + timestamp + appKey
     *
     * @param appId
     * @param appKey
     * @param timestamp
     * @return
     */
    private String generateSign(String appId,  String appKey,String timestamp) {

        try {
            return convertToHexString(MessageDigest.getInstance("MD5").digest((appId + timestamp + appKey).getBytes()));
        }catch (NoSuchAlgorithmException e) {
            return UUID.randomUUID().toString();
        }
    }

    /**
     * 转成十六进制字符串
     *
     * @param data
     * @return
     */
    private String convertToHexString(byte[] data) {

        StringBuffer sb = new StringBuffer(data.length);
        String sTemp;

        for (int i = 0; i < data.length; i++) {
            sTemp = Integer.toHexString(0xFF & data[i]);
            if (sTemp.length() < 2)
                sb.append(0);
            sb.append(sTemp.toLowerCase());
        }

        return sb.toString();
    }

    public void refreshToken() {
        if(tokenMap.get(currToken) ==null){
            System.out.println("refreshToken -- no token cache, return! ");
            return;
        }else{
            System.out.println("refreshToken -- token cached, "+ tokenMap.get(currToken));
        }

        // 构建URL
        String url = baseUrl + refreshTokenEndpoint;

        // 创建请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set("Authorization", "Bearer " + tokenMap.get(currToken).getRefreshToken());

        // 创建请求体

        String requestJson = String.format("{\"appId\":\"%s\",\"grantType\":\"refreshToken\"}", appId);

        // 封装请求实体
        HttpEntity<String> entity = new HttpEntity<>(requestJson, headers);

        try {
            ResponseEntity<ResultData<UserTokenResultResponse>> response = restTemplate.exchange(url, HttpMethod.POST, entity,
                    new ParameterizedTypeReference<ResultData<UserTokenResultResponse>>() {});
            // 如果没有抛出异常，那么可以安全地使用response对象
            System.out.println("refresh result: " + response);
            if (response.getBody() != null && response.getBody().getCode() == 0) {
                tokenMap.put("old", tokenMap.get(currToken));
                tokenMap.put(currToken, response.getBody().getData());
            } else {
                System.out.println("Refresh failed!!");
            }
        } catch (HttpClientErrorException e) {
            if (e.getStatusCode() == HttpStatus.UNAUTHORIZED) {
                // 处理401错误
                System.out.println("Unauthorized request, refreshing token or taking appropriate actions.");
                getAccessToken();
            } else {
                // 处理其他的客户端错误
                System.out.println("Client error occurred: " + e.getStatusCode());
            }
        } catch (Exception e) {
            // 处理其他异常
            System.out.println("An error occurred: " + e.getMessage());
        }

    }

    // 每3.5小时执行一次
    @Scheduled(fixedRate = 12600000)
//    @Scheduled(fixedRate = 3600000)
    public void scheduledRefreshToken() {
        System.out.println("Start refreshing token: " + LocalDateTime.now());
        refreshToken();
    }

    public Map getCache(){
        return tokenMap;
    }

    public void clearCache(){
        tokenMap.clear();
    }

}
