export const fetchJson = (url: string): Promise<any> => {
  return fetch(url).then(res => res.json());
}

type DebounceFunction = (...args: any[]) => void;

export function debounce<T extends DebounceFunction>(func: T, delay: number): T {
  let timeoutId: ReturnType<typeof setTimeout> | null;

  return function(this: ThisParameterType<T>, ...args: Parameters<T>): void {
    if (timeoutId) {
      clearTimeout(timeoutId);
    }
    timeoutId = setTimeout(() => {
      func.apply(this, args);
    }, delay);
  } as T;
}

export const isImg = (url: string): boolean => {
  const _ex: string = url.split('.').pop() || '';
  const extension = _ex.toLowerCase();
  const imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg', 'webp'];
  // const videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm', 'mkv'];

  if (imageExtensions.includes(extension)) {
    return true;
  } 
  return false;
}

export const checkAuthorization = async() => {
  try {
    if (!window.navigator.mediaDevices) {
      throw new Error('您当前浏览器或者协议暂不支持麦克风');
    }
    await window.navigator.mediaDevices.getUserMedia({ 'audio': true });
  } catch(e: any) {
    let msg = e.message;
    switch (e.name || e.message) {
      case "PERMISSION_DENIED":
      case "PermissionDeniedError":
      case "NotAllowedError":
        msg = "用户拒绝提供权限";
        break;
      case "NOT_SUPPORTED_ERROR":
      case "NotSupportedError":
        msg = "浏览器不支持您当前选择的设备";
        break;
      case "MANDATORY_UNSATISFIED_ERROR":
      case "MandatoryUnsatisfiedError":
        msg = "media_mandatory_error";
        break;
      default:
        msg = `无法打开麦克风,原因：${e.code || e.name}`
    }

    throw new Error(msg);
  }
}

